"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.systemRouter = void 0;
const systemController_1 = require("../controllers/systemController");
const auth_1 = require("../middleware/auth");
const settingController_1 = require("../controllers/settingController");
const express = require("express");
const multer = require("multer");
const fs = require("fs");
const path = require("path");
const router = express.Router();
// Configure multer storage for system images
// Fix the multer destination to match your static serving
const storage = multer.diskStorage({
    destination: (req, file, cb) => {
        const folder = "/uploads/system";
        const dir = path.join(__dirname, `../${folder}`);
        fs.mkdir(dir, { recursive: true }, (err) => {
            if (err) {
                console.error("Error creating upload directory:", err);
                return cb(new Error(`Failed to create upload directory: ${err.message}`), "");
            }
            cb(null, dir);
        });
    },
    filename: (req, file, cb) => {
        const uniqueName = `${Date.now()}_${file.originalname}`;
        cb(null, uniqueName);
    },
});
const upload = multer({ storage });
// ✅ Define routes BEFORE exporting
router.get("/setting", systemController_1.systemController.getSetting);
router.get("/banks", auth_1.authenticate, systemController_1.systemController.getBanks);
router.get("/all", auth_1.authenticate, systemController_1.systemController.getVip);
router.get("/taskInfo", systemController_1.systemController.getTaskInfo);
// Bank routes
router.get("/banks", upload.fields([{ name: "logo", maxCount: 1 }]), settingController_1.bankController.getAllBanks);
router.post("/banks", upload.fields([{ name: "logo", maxCount: 1 }]), settingController_1.bankController.createBank);
router.put("/banks/:id", upload.fields([{ name: "logo", maxCount: 1 }]), settingController_1.bankController.updateBank);
router.delete("/banks/:id", settingController_1.bankController.deleteBank);
// Settings routes
router.get("/settings", settingController_1.settingController.getSettings);
router.post("/settings", settingController_1.settingController.updateSettings);
// TaskInfo routes with file upload
router.get("/task-info", settingController_1.taskInfoController.getAllTaskInfo);
router.post("/task-info", upload.fields([{ name: "logo", maxCount: 1 }]), settingController_1.taskInfoController.createTaskInfo);
router.put("/task-info/:id", upload.fields([{ name: "logo", maxCount: 1 }]), settingController_1.taskInfoController.updateTaskInfo);
router.delete("/task-info/:id", settingController_1.taskInfoController.deleteTaskInfo);
// VIP routes with file upload
router.get("/vips", settingController_1.vipController.getAllVips);
router.post("/vips", upload.fields([{ name: "logo", maxCount: 1 }]), settingController_1.vipController.createVip);
router.put("/vips/:id", upload.fields([{ name: "logo", maxCount: 1 }]), settingController_1.vipController.updateVip);
router.delete("/vips/:id", settingController_1.vipController.deleteVip);
exports.systemRouter = router;
