"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = __importDefault(require("express"));
const cors_1 = __importDefault(require("cors"));
const path_1 = __importDefault(require("path"));
const helmet_1 = __importDefault(require("helmet"));
const compression_1 = __importDefault(require("compression"));
const express_rate_limit_1 = __importDefault(require("express-rate-limit"));
const hpp_1 = __importDefault(require("hpp"));
const index_1 = __importDefault(require("./routes/index"));
const app = (0, express_1.default)();
const port = process.env.PORT || 3000;
// Security Middlewares
app.use((0, helmet_1.default)({
    crossOriginResourcePolicy: { policy: "same-site" },
    contentSecurityPolicy: {
        directives: {
            defaultSrc: ["'self'"],
            styleSrc: ["'self'", "'unsafe-inline'"],
            scriptSrc: ["'self'"],
            imgSrc: ["'self'", "data:", "blob:", "https:"],
            connectSrc: ["'self'"],
        },
    },
}));
app.use((req, res, next) => {
    res.header("Access-Control-Allow-Origin", "*");
    res.header("Access-Control-Allow-Methods", "GET, POST, PUT, DELETE, OPTIONS");
    res.header("Access-Control-Allow-Headers", "Content-Type, Authorization");
    // Security headers for mobile
    res.header("Strict-Transport-Security", "max-age=31536000; includeSubDomains");
    res.header("X-Content-Type-Options", "nosniff");
    res.header("X-Frame-Options", "DENY");
    next();
});
// Rate limiting
const generalLimiter = (0, express_rate_limit_1.default)({
    windowMs: 15 * 60 * 1000,
    max: process.env.NODE_ENV === "NODE_ENV" ? 500 : 1000,
    message: {
        error: "Too many requests from this IP, please try again later.",
    },
    standardHeaders: true,
    legacyHeaders: false,
});
const uploadLimiter = (0, express_rate_limit_1.default)({
    windowMs: 15 * 60 * 1000,
    max: 30,
    message: {
        error: "Too many upload requests, please try again later.",
    },
});
// Apply general rate limiting to all requests
app.use(generalLimiter);
// CORS configuration
const corsOptions = {
    origin: process.env.FRONTEND_URL
        ? process.env.FRONTEND_URL.split(",")
        : ["http://localhost:3000", "http://localhost:3001"],
    credentials: true,
    methods: ["GET", "POST", "PUT", "DELETE", "PATCH"],
    allowedHeaders: ["Content-Type", "Authorization", "X-Requested-With"],
};
app.use((0, cors_1.default)(corsOptions));
// Body parsing middleware
app.use(express_1.default.json({
    limit: "10mb",
}));
app.use(express_1.default.urlencoded({
    extended: true,
    limit: "10mb",
}));
// Prevent parameter pollution
app.use((0, hpp_1.default)());
// Compression middleware
app.use((0, compression_1.default)());
// Static files with security headers
app.use("/uploads", uploadLimiter, express_1.default.static(path_1.default.join(__dirname, "uploads"), {
    // express.static(path.join(__dirname, "uploads"), {
    setHeaders: (res, filePath) => {
        res.set("Cross-Origin-Resource-Policy", "same-site");
        res.set("X-Content-Type-Options", "nosniff");
        res.set("X-Frame-Options", "DENY");
        if (filePath.match(/\.(jpg|jpeg|png|gif|webp|avif)$/i)) {
            res.set("Cache-Control", "public, max-age=86400");
        }
        else if (filePath.match(/\.(js|css)$/)) {
            res.set("Cache-Control", "public, max-age=31536000");
        }
        else {
            res.set("Cache-Control", "public, max-age=3600");
        }
    },
    index: false,
    dotfiles: "deny",
}));
app.use(express_1.default.static("public"));
// Request logging middleware
// app.use((req, res, next) => {
//   const start = Date.now();
//   res.on("finish", () => {
//     const duration = Date.now() - start;
//     console.log(
//       `${req.method} ${req.originalUrl} ${res.statusCode} ${duration}ms`
//     );
//   });
//   next();
// });
// API routes
app.use("/api", index_1.default);
// Health check endpoint
app.get("/health", (req, res) => {
    res.status(200).json({
        status: "OK",
        timestamp: new Date().toISOString(),
        uptime: process.uptime(),
        memory: process.memoryUsage(),
        environment: process.env.NODE_ENV || "development",
    });
});
// ✅ FIXED: Proper 404 handler - Place this AFTER all other routes
app.use((req, res) => {
    res.status(404).json({
        error: "Route not found",
        path: req.originalUrl,
        method: req.method,
        timestamp: new Date().toISOString(),
    });
});
// Global error handling middleware
app.use((error, req, res, next) => {
    console.error("Error:", {
        message: error.message,
        url: req.originalUrl,
        method: req.method,
        timestamp: new Date().toISOString(),
    });
    // Prisma errors
    if (error.code) {
        switch (error.code) {
            case "P2002":
                return res.status(409).json({
                    error: "Conflict",
                    message: "Record already exists",
                });
            case "P2025":
                return res.status(404).json({
                    error: "Not Found",
                    message: "Record not found",
                });
        }
    }
    const statusCode = error.statusCode || 500;
    const isProduction = process.env.NODE_ENV === "production";
    res.status(statusCode).json({
        error: isProduction && statusCode === 500
            ? "Internal server error"
            : error.message,
        ...(!isProduction && { stack: error.stack }),
    });
});
// Graceful shutdown
const gracefulShutdown = (signal) => {
    console.log(`\n${signal} received, shutting down gracefully`);
    server.close(() => {
        console.log("HTTP server closed");
        process.exit(0);
    });
    setTimeout(() => {
        console.error("Forcing shutdown after timeout");
        process.exit(1);
    }, 10000);
};
process.on("SIGTERM", () => gracefulShutdown("SIGTERM"));
process.on("SIGINT", () => gracefulShutdown("SIGINT"));
// Start server
const server = app.listen(port, () => {
    console.log(`
🚀 Server running in ${process.env.NODE_ENV || "development"} mode
📍 Port: ${port}
📅 Started: ${new Date().toISOString()}
  `);
});
exports.default = app;
